#version 120

varying vec2 uv;
varying vec3 light_vector;
varying vec3 eye_vector;
varying vec3 vert_position;
varying vec3 normal_vector;

uniform sampler2D textureSampler;
uniform sampler2D normalSampler;
uniform sampler2D specularSampler;
uniform samplerCube skyboxSampler;
uniform sampler2D skyFrontSampler;
uniform sampler2D skyBackSampler;

uniform vec4 ambient_color;
uniform vec3 sun_direction;
uniform vec4 sun_color;
uniform int specular_exponent;
uniform vec3 camera_position;
uniform float reflectivity;
uniform vec2 textureScale;
uniform int time;

//Because radians
#define PI 3.14159265
#define TWO_PI 6.2831853

//How long the reflected sky sphere takes to complete one full rotation
#define ROTATION_TIME 60000

//Color added at the end to lighten the ice
#define ICE_COLOR_ADD vec4(0.2, 0.25, 0.3, 1.0)

void main() {
	//Correct UV based on texture scale
	vec2 scaled_uv = uv * textureScale;
	
	//Texture values
	vec3 material_color = texture2D(textureSampler, scaled_uv).rgb;
	vec3 specular_color = texture2D(specularSampler, scaled_uv).rgb;
	vec3 normal_color = normalize(texture2D(normalSampler, scaled_uv).rgb * 2.0 - 1.0);

	//Normalize the light vector so we can dot it
	vec3 light_normal = normalize(light_vector);

	//Cosine of the angle from the light to the normal
	float cosTheta = clamp(dot(normal_color, light_normal), 0, 0.6);

	//Opposite cosine so that sides completely in the dark are still illuminated slightly.
	// Otherwise the textures on the undersides have no bump mapping and look gross.
	float cosNtheta = clamp(dot(-normal_color, light_normal), 0.8, 1.0) + 0.1;

	//Direction of the light reflection
	vec3 light_reflection = reflect(-light_normal, normal_color);

	//Angle from the eye vector and reflect vector
	float cosAlpha = clamp(dot(normalize(eye_vector), light_reflection), 0, 1.0);

	//Diffuse color
	gl_FragColor = vec4(material_color * (sun_color.rgb * sun_color.a), 1);

	//Bump mapping
	gl_FragColor *= cosTheta;

	//Ambient color
	gl_FragColor += vec4(material_color * ambient_color.rgb * cosNtheta, 1);

	//Worldspace normal taking normal mapping into account
	vec3 normal_model = normalize(reflect(-normal_vector, normal_color));
	//Direction from camera to vertex
	vec3 camera_direction = normalize(vert_position - camera_position);
	//Reflect the camera off the normal so we know where on the skysphere to show
	vec3 camera_reflection = reflect(camera_direction, normal_model);

	//Find UV coordinates of the reflection on the sky sphere
	vec2 skyUV = vec2(0.5 - (atan(camera_reflection.y, camera_reflection.x) / TWO_PI), 0.5 - (asin(camera_reflection.z) / PI));

	//Rotate the coordinates so the clouds spin around
	skyUV += vec2((float(time) / ROTATION_TIME), 0);
	//Cloud color from the skysphere
	vec4 reflectionBackColor = texture2D(skyBackSampler, skyUV);
	vec4 reflectionFrontColor = texture2D(skyFrontSampler, skyUV);

	//Let the front color overlay the back
	vec4 reflectionColor = mix(reflectionFrontColor, reflectionBackColor, reflectionBackColor.a);

	//Apply the reflected skysphere color
	gl_FragColor = mix(gl_FragColor, reflectionColor, reflectivity);

	//Ice is too dark without a little color added, so we adjust it a bit
	gl_FragColor += ICE_COLOR_ADD * cosTheta;

	//Specular highlights
	gl_FragColor += vec4(specular_color * (sun_color.rgb * sun_color.a) * pow(cosAlpha, specular_exponent), 1);

	// gl_FragColor = vec4(skyUV, 0, 1);

	// gl_FragColor = vec4((normal_model + 1) / 2, 1);

	// gl_FragColor = vec4((normal_vector + 1) / 2, 1);

	// gl_FragColor = vec4(R, 1);

	// gl_FragColor = vec4(cosTheta);

	// gl_FragColor = vec4(eye_vector, 1);

	// gl_FragColor = vec4(1, 0, 1, 1);
}