//
// scaletozero.cs
//
// Defines the Scale To Zero tool, which is equivelent to using the scale tool, and entering 0 for the selected axes. It is
// perticularly useful for combining verticies to make nice curves.
//
// The instance makes use of the following dynamic fields:
// static    - Points to the static ScriptObject
// dirty     - Flag to indicate that the tool needs to be refreshed on screen
// active    - Flag to indicate that the tool is active, draw its handles, and interact with the user
// update    - Store a value to be returned to Constructor when asked about the tool's edit state, such as do nothing or update with new settings, etc.

// Revision History:
// June 21, 2009		ShadowMarble		Created script file
//

package ScaleToZero
{
   //************************************************************************************
   //*** Standard Tool Functions
   //************************************************************************************
   
   //************************************************************************************
   // Activate()
   //
   // Called when the tool is activated.  %version holds the current version of this
   // tool type in Constructor to allow the tool to step down features if required.
   // %inst is actually a ScriptObject behind the scenes that allows for the tool's
   // instance to be attached to it -- which is typically a ScriptObject itself.
   // %static is a ScriptObject that allows anything to be attached to it that will
   // presist across tool instances.
   // Return a Tool Return Function to indicate success of the tool's activation.
   function Plugin::Activate(%this, %version, %inst, %static)
   {
      //error("ScaleToZero: Activate(" @ %version @ "," @ %inst @ "," @ %static @ ")");

      //*** Check for a valid version
      if(%version != 1)
      {
         return tool.FUNC_BADVERSION();
      }

      //*** Build the tool's instance
      %plugin = new ScriptObject();
      
      //*** This plugin doesn't make use of static data but we still need to
      //*** assign it.
      %plugin.static = %static;

      //*** Setup some additional attributes for the instance
      %plugin.dirty = tool.DIRTY_NONE();
      %plugin.active = true;
      %plugin.update = tool.EDIT_DONOTHING();
      //ScaleToZero_Reset(%plugin);
      
      //*** Pass along the instance
      %inst.instance = %plugin;
      %inst.flagsInterface = tool.IFLAG_STANDARDGEOMETRY() + tool.IFLAG_DRAWALLSAME(); // Set up the tool with the standard geometry creation GUI and all views to use the same drawing buffer.  Our Draw() will only be called once.
      %inst.flagsApply = tool.AFLAG_NONE();
      
      //*** Return that everything is OK
      return tool.FUNC_OK();
   }
   
   //************************************************************************************
   // Done()
   //
   // Called when the user is finished with the tool.  Typically any allocated
   // memory is freed here.  %inst and %static are the same as those in the
   // activation function.  This function does not return a value.
   function Plugin::Done(%this, %inst, %static)
   {
      //error("ScaleToZero: Done(" @ %inst @ "," @ %static @ ")");

      %plugin = %inst.instance;
      
      if(%plugin)
      {         
         //*** Delete our instance
         %plugin.delete();
         %inst.instance = 0;
      }
   }

   //************************************************************************************
   // Dirty()
   //
   // This function is called to determine if the tool needs to be redrawn.  Return a
   // combination of the tool.DIRTY_* flags added together to indicate that the tool's
   // features (but not geometry) should be redrawn.
   function Plugin::Dirty(%this, %inst)
   {
      //error("ScaleToZero: Dirty(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.dirty ? tool.DIRTY_APPEARANCE() : tool.DIRTY_NONE();
   }

   //************************************************************************************
   // Draw()
   //
   // This function is called to draw the tool itself.  Geometry is not built here but
   // in BuildGeometry().  The %draw parameter points to the ToolDrawing class
   // and is used to build up the tool's wireframe.  Just before this function is called,
   // Constructor will clear the draw buffer, so the tool is responsible for recreating
   // the tool's appearance.  This function may be called multiple times, once for each
   // view type.  The %draw.getView(); function returns the current view type.  The tool
   // is not required to do anything different for each view type and may send the same drawing
   // commands on each call to this function, although it may be wise to treat the UV view
   // as a special case.  Draw() does not return a value.
   function Plugin::Draw(%this, %inst, %draw)
   {
      //error("ScaleToZero: Draw(" @ %inst @ "," @ %draw @ ")");

      %plugin = %inst.instance;
      
      //*** If the tool is not active, then don't draw it
      if(!%plugin.active)
         return;

      //*** Indicate that we've drawn the tool
      %plugin.dirty = tool.DIRTY_NONE();
   }

   //************************************************************************************
   // CheckEditAction()
   //
   // This function is called to determine how to handle the tool's geometry.  Return
   // one of the tool.EDIT_* flags to indicate how to modify the geometry based on the
   // latest change.
   function Plugin::CheckEditAction(%this, %inst)
   {
      //error("ScaleToZero: CheckEditAction(" @ %inst @ ")");

      %plugin = %inst.instance;
      
      return %plugin.update;
   }

   //************************************************************************************
   // EndEditAction()
   //
   // This function is called after the completion of a mouse down to mouse drag to mouse
   // up sequence.  This may be called a number of times.  The %keep parameter is set
   // based on what is returned from the CheckEditAction() function.  This function
   // does not return a value.
   function Plugin::EndEditAction(%this, %inst, %keep)
   {
      //error("ScaleToZero: EndEditAction(" @ %inst @ "," @ %keep @ ")");

      %plugin = %inst.instance;
      
      //*** If we're to keep the operation, then trigger the work
      if(%keep)
      {
         //*** Perform the hollow by forcing an update
         %plugin.update = tool.EDIT_UPDATE();
         %plugin.dirty = tool.DIRTY_APPEARANCE();
         tool.refresh();
         
         //scene.notifyBrushRefresh();
      }
      
      %plugin.update = tool.EDIT_DONOTHING();
      %plugin.active = false;

   }

   //************************************************************************************
   // BuildGeometry()
   //
   // This function is called to build/edit the tool's actual geometry.  %edit points to
   // the geometry edit operation structure.  Return a Tool Return Function to indicate
   // success of the tool's operation on the geometry.
   function Plugin::BuildGeometry(%this, %inst, %edit)
   {
      //error("ScaleToZero: BuildGeometry(" @ %inst @ "," @ %edit @ ")");

      %plugin = %inst.instance;

      //*** Scale the geometry if we're active.
      if(%plugin.active)
      {
         %offset = !%plugin.static.axis[0] SPC !%plugin.static.axis[1] SPC !%plugin.static.axis[2];

         tool.setUndoEntryLabel("Scale To Zero (" @ %offset @ ")");

         %edit.scaleItemsCenterRelBuf(%offset, select.getSelectionCenter());

         %edit.updateSelectionBounds();
      }
      
      //*** As we've now worked on the geometry, set the edit update indicator to do nothing.
      %plugin.update = tool.EDIT_DONOTHING();
      
      return tool.FUNC_OK();
   }

   //************************************************************************************
   // UserEvent()
   //
   // This function is called when the user does something to the tool, such as activate
   // it or reset it.  %userevent is the event that the user caused.  This function does
   // not return a value.
   function Plugin::UserEvent(%this, %inst, %userevent)
   {
      //error("ScaleToZero: UserEvent(" @ %inst @ "," @ %userevent @ ")");

      %plugin = %inst.instance;
      
      switch(%userevent)
      {
         //*** User activated the tool such that we should continue to use the current
         //*** settings (ie: same centre and size).  This is different from the user
         //*** clicking in the 3D interface to draw a new object.
         case tool.EVENT_ACTIVATE():
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
            
         //*** The user has asked that the tool be reset back to its default values/settings.
         case tool.EVENT_RESET():
            ScaleToZero_Reset(%plugin);
            %plugin.update = tool.EDIT_UPDATE();
            %plugin.active = true;
            %plugin.dirty = tool.DIRTY_APPEARANCE();
         
         //*** The user has deactivated the tool.  If the tool is active, then we tell
         //*** Constructor to reject any interactive action that is partly complete.  This
         //*** will discard any geometry the tool has created.
         case tool.EVENT_DEACTIVATE():
            if(%plugin.active)
            {
               %plugin.update = tool.EDIT_REJECT();
            }
            %plugin.dirty = tool.DIRTY_APPEARANCE();
      }
   }

   //************************************************************************************
   // Interface()
   //
   // This function sets up the GUI for the tool to allow the user to change the tool's
   // parameters.  %form points to the interface construction class that this function
   // makes calls to when building the interface.  This function does not return a value.
   function Plugin::Interface(%this, %inst, %form)
   {
      //error("ScaleToZero: Interface(" @ %inst @ "," @ %form @ ")");

      %plugin = %inst.instance;

      //*** Provide a title
      %form.defineTitle("Scale to Zero");

      //*** Add our fields to the form in the order we wish them displayed.  A field
      //*** with an ID of -1 will not have a value get/set.
      %form.addField( 0, "Axis X"   ,"checkbox");
      %form.addField( 1, "Y"        ,"checkbox");
      %form.addField( 2, "Z"        ,"checkbox");
   }

   //************************************************************************************
   // InterfaceGet()
   //
   // This function is called to retrieve a value from the tool given the field's ID
   // in %id.  The value of the field is then returned.
   function Plugin::InterfaceGet(%this, %inst, %id)
   {
      //error("TransformRotate: InterfaceGet(" @ %inst @ "," @ %id @ ")");

      %plugin = %inst.instance;
      
      return %plugin.static.axis[%id];
   }

   //************************************************************************************
   // InterfaceSet()
   //
   // This function is called to set a value for the tool given the field's ID
   // in %id, and the value to set to in %value.  This function returns tool.FUNC_OK();
   // if the value was accepted.  Otherwise it returns tool.FUNC_BADVALUE(); to indicate
   // that the given value is invalid and the correct value should be retrieved from the
   // tool once again.
   function Plugin::InterfaceSet(%this, %inst, %id, %value)
   {
      error("ScaleToZero: InterfaceSet(" @ %inst @ "," @ %id @ "," @ %value @")");

      %plugin = %inst.instance;
      
      %plugin.static.axis[%id] = %value < 0 ? 0 : %value;
      
      return tool.FUNC_OK();
   }
   
      
   //************************************************************************************
   //*** Tool Specific Functions
   //************************************************************************************
   
   //*** Reset the transform instance to default values
   function ScaleToZero_Reset(%plugin, %angleonly)
   {
      %plugin.static.axis[0] = 0;
      %plugin.static.axis[1] = 0;
      %plugin.static.axis[2] = 0;
   }
   
};

tool.register("ScaleToZero", tool.typeInteractive(), tool.RFLAG_STORETRANSFORMS() + tool.RFLAG_ALLOWNONMODALSELECTION(), "Scale To Zero" );
tool.setToolProperty("ScaleToZero", "Icon", "scaletozero");
tool.setToolProperty("ScaleToZero", "Group", "Most Useful Tools Ever");
